<?php
// auth.php — protect pages, enforce idle + absolute timeouts, safe redirects
declare(strict_types=1);
require_once __DIR__ . "/../helpers/helpers.php";

/* -------------------- Config -------------------- */
// Define once (won’t redeclare if included multiple times)
if (!defined('ADMIN_BASE')) define('ADMIN_BASE', '/admin-panel/');  // MUST start & end with /
if (!defined('LOGIN_PATH')) define('LOGIN_PATH', ADMIN_BASE . 'auth/login.php');

const IDLE_TIMEOUT_SEC = 3600;   // 1 hour inactivity
const ABS_TIMEOUT_SEC  = 8 * 3600; // 8 hours absolute lifetime

/* -------------------- Session (secure flags) -------------------- */
$isHttps = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off')
        || (($_SERVER['HTTP_X_FORWARDED_PROTO'] ?? '') === 'https');

if (session_status() !== PHP_SESSION_ACTIVE) {
  session_name('homeland_admin');
  session_set_cookie_params([
    'lifetime' => 0,
    'path'     => '/',
    'domain'   => '',
    'secure'   => $isHttps,
    'httponly' => true,
    'samesite' => 'Strict',
  ]);
  session_start();
}

/* -------------------- Helpers -------------------- */
function kill_session(): void {
  $_SESSION = [];
  if (ini_get('session.use_cookies')) {
    $p = session_get_cookie_params();
    setcookie(session_name(), '', time() - 42000, $p['path'], $p['domain'], $p['secure'], $p['httponly']);
  }
  session_destroy();
}

/**
 * Allow only absolute paths under ADMIN_BASE (block full URLs and outside paths).
 */


/* -------------------- Guard & Timeouts -------------------- */
$now = time();

// Normalize current request path (no query) for comparisons
$reqUri  = (string)($_SERVER['REQUEST_URI'] ?? ADMIN_BASE); // keeps query for next=
$reqPath = parse_url($reqUri, PHP_URL_PATH) ?: ADMIN_BASE;

// Are we on the login page?
$onLogin = ($reqPath === LOGIN_PATH);

// If not logged in and not already on login → send to login ONCE with next=<original>
if (empty($_SESSION['admin_id']) && !$onLogin) {
  $next = rawurlencode($reqUri); // encode once; do not re-encode later
  header('Location: ' . LOGIN_PATH . '?next=' . $next);
  exit;
}

// If logged in, enforce timeouts
if (!empty($_SESSION['admin_id'])) {
  $_SESSION['_created_at']     = $_SESSION['_created_at']     ?? $now;
  $_SESSION['_last_activity']  = $_SESSION['_last_activity']  ?? $now;

  // Absolute session cap
  if (($now - (int)$_SESSION['_created_at']) > ABS_TIMEOUT_SEC) {
    kill_session();
    header('Location: ' . LOGIN_PATH . '?msg=expired');
    exit;
  }

  // Idle timeout
  if (($now - (int)$_SESSION['_last_activity']) > IDLE_TIMEOUT_SEC) {
    kill_session();
    header('Location: ' . LOGIN_PATH . '?msg=timeout');
    exit;
  }

  // Refresh activity timestamp
  $_SESSION['_last_activity'] = $now;

  // Periodically rotate session ID (limits fixation window)
  if (!isset($_SESSION['_rotated_at']) || ($now - (int)$_SESSION['_rotated_at']) > 900) { // 15 min
    session_regenerate_id(true);
    $_SESSION['_rotated_at'] = $now;
  }
}
