<?php
declare(strict_types=1);

/* ---------------- Security headers ---------------- */
header('X-Frame-Options: DENY');
header('X-Content-Type-Options: nosniff');
header('Referrer-Policy: strict-origin-when-cross-origin');
header('Permissions-Policy: camera=(), microphone=(), geolocation=()'); // adjust if needed

require_once __DIR__ . '/../../functions/helpers.php';   // unified helpers
require_once __DIR__ . '/../../functions/database.php';  // your PDO

/* ---------------- Config (must match auth.php) ---------------- */
if (!defined('ADMIN_BASE')) define('ADMIN_BASE', '/admin-panel/');         // MUST start & end with /
if (!defined('LOGIN_PATH')) define('LOGIN_PATH', ADMIN_BASE . 'auth/login.php');    // absolute URL path
$assetBase = ADMINURL; // absolute base URL for assets

/* ---------------- Session (secure flags) ---------------- */
$isHttps = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off')
        || (($_SERVER['HTTP_X_FORWARDED_PROTO'] ?? '') === 'https');

if (session_status() !== PHP_SESSION_ACTIVE) {
  session_name('homeland_admin');
  session_set_cookie_params([
    'lifetime' => 0,
    'path'     => '/',
    'domain'   => '',
    'secure'   => $isHttps,
    'httponly' => true,
    'samesite' => 'Strict',
  ]);
  session_start();
}

/* ---------------- Early redirect if already logged in ---------------- */
if (!empty($_SESSION['admin_id'])) {
  // respect ?next= if valid (and not the login page), else go to dashboard
  $next = $_GET['next'] ?? '';
  $to = ADMIN_BASE;
  if ($next && is_same_origin_path($next) && $next !== LOGIN_PATH) {
    $to = $next;
  }
  header('Location: ' . $to);
  exit;
}

/* ---------------- DB helpers (PDO expected in database.php) ---------------- */
if (!function_exists('db_find_admin_by_email')) {
  function db_find_admin_by_email(string $email): ?array {
    global $pdo;
    if (!isset($pdo) || !($pdo instanceof PDO)) {
      throw new RuntimeException('PDO connection $pdo not available');
    }
    $stmt = $pdo->prepare("SELECT id, email, password AS pwd FROM admins WHERE email = :email LIMIT 1");
    $stmt->execute([':email' => $email]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    return $row ?: null;
  }
}
if (!function_exists('db_update_admin_password_hash')) {
  function db_update_admin_password_hash(int $id, string $newHash): void {
    global $pdo;
    $stmt = $pdo->prepare("UPDATE admins SET password = :pwd WHERE id = :id");
    $stmt->execute([':pwd' => $newHash, ':id' => $id]);
  }
}
if (!function_exists('audit_login')) {
  function audit_login(string $email, bool $ok): void {
    try {
      global $pdo;
      if (!isset($pdo) || !($pdo instanceof PDO)) return;
      $stmt = $pdo->prepare("INSERT INTO login_audit (email, ip, ok, ts) VALUES (:email, :ip, :ok, NOW())");
      $stmt->execute([
        ':email' => $email,
        ':ip'    => $_SERVER['REMOTE_ADDR'] ?? '',
        ':ok'    => $ok ? 1 : 0,
      ]);
    } catch (Throwable $e) { /* noop */ }
  }
}

/* ---------------- CSRF ---------------- */
if (empty($_SESSION['csrf'])) {
  $_SESSION['csrf'] = bin2hex(random_bytes(32));
}

/* ---------------- Basic rate limit ---------------- */
$ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
$_SESSION['login_attempts']     = $_SESSION['login_attempts']     ?? [];
$_SESSION['login_block_until']  = $_SESSION['login_block_until']  ?? 0;

$WINDOW_SEC   = 15 * 60;   // 15 minutes window
$MAX_ATTEMPTS = 5;         // 5 attempts per window
$BLOCK_SEC    = 10 * 60;   // block for 10 minutes after max
$now          = time();

$attemptRec   = $_SESSION['login_attempts'][$ip] ?? ['count' => 0, 'start' => $now];
if (($now - $attemptRec['start']) > $WINDOW_SEC) {
  $attemptRec = ['count' => 0, 'start' => $now];
}
$blocked = $now < (int)($_SESSION['login_block_until'] ?? 0);

/* ---------------- Handle POST ---------------- */
$errors = ['email'=>null, 'password'=>null, 'general'=>null];
$email_value = '';
$show_block_msg = $blocked;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  // CSRF check first
  $postedCsrf = $_POST['csrf'] ?? '';
  if (!hash_equals($_SESSION['csrf'], $postedCsrf)) {
    $errors['general'] = 'Something went wrong. Please refresh and try again.';
  }
  elseif ($blocked) {
    $errors['general'] = 'Too many attempts. Please try again later.';
  } else {
    // Inputs
    $email    = trim((string)($_POST['email'] ?? ''));
    $password = (string)($_POST['password'] ?? '');
    $email_value = $email;

    // Validation
    if ($email === '') {
      $errors['email'] = "Email is required";
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
      $errors['email'] = "Invalid email format";
    }
    if ($password === '') {
      $errors['password'] = "Password is required";
    }

    if (empty(array_filter($errors))) {
      $admin  = db_find_admin_by_email($email);
      $authOk = false;

      if ($admin) {
        $stored = (string)$admin['pwd'];

        // Modern hashes (bcrypt/argon2)
        if (preg_match('/^\$2y\$/', $stored) || str_starts_with($stored, '$argon2')) {
          $authOk = password_verify($password, $stored);
          if ($authOk && password_needs_rehash($stored, PASSWORD_DEFAULT)) {
            db_update_admin_password_hash((int)$admin['id'], password_hash($password, PASSWORD_DEFAULT));
          }
        }
        // Legacy sha256 hex → migrate on success
        elseif (preg_match('/^[a-f0-9]{64}$/i', $stored)) {
          $candidate = hash('sha256', $password);
          $authOk = hash_equals(strtolower($stored), strtolower($candidate));
          if ($authOk) {
            db_update_admin_password_hash((int)$admin['id'], password_hash($password, PASSWORD_DEFAULT));
          }
        }
      }

      if ($authOk) {
        // success
        $_SESSION['login_attempts'][$ip] = ['count' => 0, 'start' => time()];
        $_SESSION['login_block_until']   = 0;

        session_regenerate_id(true);
        $_SESSION['admin_id']     = (int)$admin['id'];
        $_SESSION['admin_email']  = (string)$admin['email'];
        $_SESSION['_created_at']  = $_SESSION['_last_activity'] = $_SESSION['_rotated_at'] = time();
        unset($_SESSION['csrf']); // one-time token

        audit_login($email, true);

        // Safe post-login redirect
        $to   = ADMIN_BASE;                // default: dashboard root
        $next = $_GET['next'] ?? '';       // NEVER re-encode; came urlencoded from auth.php
        if ($next && is_same_origin_path($next) && $next !== LOGIN_PATH) {
          $to = $next;
        }
        header('Location: ' . $to);
        exit;
      } else {
        // failure: bump attempts and maybe block
        $attemptRec['count']++;
        if ($attemptRec['count'] >= $MAX_ATTEMPTS) {
          $_SESSION['login_block_until'] = $now + $BLOCK_SEC;
          $show_block_msg = true;
        }
        $_SESSION['login_attempts'][$ip] = $attemptRec;

        $errors['general'] = "Email or password is incorrect";
        audit_login($email, false);
      }
    }
  }
}

$email_err = $errors['email'] ?: null;
$pass_err  = $errors['password'] ?: null;
$gen_err   = $errors['general'] ?: null;
?>
<!DOCTYPE html>
<html lang="en">
  <head>
    <meta charset="utf-8" />
    <meta http-equiv="X-UA-Compatible" content="IE=edge" />
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no" />
    <title>Login - Homeland Admin</title>
    <link href="<?= h($assetBase) ?>assets/css/styles.css" rel="stylesheet" />
    <script src="https://use.fontawesome.com/releases/v6.3.0/js/all.js" crossorigin="anonymous"></script>
    <style> body{ background:#EEE; } </style>
  </head>
  <body>
    <div id="layoutAuthentication">
      <div id="layoutAuthentication_content">
        <main>
          <div class="container">
            <div class="row justify-content-center">
              <div class="col-lg-5">
                <div class="card shadow-lg border-0 rounded-lg mt-5">
                  <div class="card-header">
                    <h3 class="text-center font-weight-light my-4">
                      Hello <span class="bg-primary text-white px-2 font-weight-bold rounded-2">Admin!</span>
                    </h3>
                    <?php if ($gen_err): ?>
                      <div class="alert alert-danger my-2" role="alert"><?= h($gen_err) ?></div>
                    <?php elseif ($show_block_msg): ?>
                      <div class="alert alert-warning my-2" role="alert">Too many attempts. Please try again later.</div>
                    <?php endif; ?>
                  </div>
                  <div class="card-body">
                    <form method="POST" autocomplete="off" novalidate>
                      <input type="hidden" name="csrf" value="<?= h($_SESSION['csrf']) ?>">

                      <div class="form-floating">
                        <input
                          class="form-control <?= $email_err ? 'is-invalid' : '' ?>"
                          id="inputEmail"
                          type="email"
                          name="email"
                          value="<?= h($email_value) ?>"
                          placeholder="name@example.com"
                          required
                          maxlength="254"
                        />
                        <label for="inputEmail">Email address</label>
                        <?php if ($email_err): ?>
                          <div class="invalid-feedback d-block"><?= h($email_err) ?></div>
                        <?php endif; ?>
                      </div>

                      <div class="form-floating mt-3">
                        <input
                          class="form-control <?= $pass_err ? 'is-invalid' : '' ?>"
                          id="inputPassword"
                          type="password"
                          name="password"
                          placeholder="Password"
                          required
                          minlength="8"
                          autocomplete="current-password"
                        />
                        <label for="inputPassword">Password</label>
                        <?php if ($pass_err): ?>
                          <div class="invalid-feedback d-block"><?= h($pass_err) ?></div>
                        <?php endif; ?>
                      </div>

                      <!-- Remember-me disabled; implement selector+token if needed -->
                      <div class="form-check mt-3">
                        <input class="form-check-input" id="inputRememberPassword" type="checkbox" value="1" disabled />
                        <label class="form-check-label" for="inputRememberPassword">Remember Password (disabled)</label>
                      </div>

                      <div class="d-flex align-items-center justify-content-between mt-4 mb-0">
                        <a class="small" href="<?= h(dirname(LOGIN_PATH)) ?>/forget_password.php">Forgot Password?</a>
                        <button type="submit" name="submit" class="btn btn-primary">Login</button>
                      </div>
                    </form>
                  </div>
                </div> <!-- card -->
              </div>
            </div>
          </div>
        </main>
      </div>
      <div id="layoutAuthentication_footer">
        <footer class="py-4 bg-light mt-auto">
          <div class="container-fluid px-4">
            <div class="d-flex align-items-center justify-content-between small">
              <div class="text-muted">Copyright &copy; Your Website 2025</div>
              <div>
                <a href="#">Privacy Policy</a>
                &middot;
                <a href="#">Terms &amp; Conditions</a>
              </div>
            </div>
          </div>
        </footer>
      </div>
    </div>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js" crossorigin="anonymous"></script>
    <script src="<?= h($assetBase) ?>js/scripts.js"></script>
  </body>
</html>
