<?php
declare(strict_types=1);

/**
 * --------- Admin base path & URL constants ----------
 * ADMIN_BASE  -> URL path (must start & end with '/'), e.g. '/Homeland/admin-panel/'
 * ADMINURL    -> absolute URL base (scheme + host + ADMIN_BASE), e.g. 'http://localhost/Homeland/admin-panel/'
 * ADMINAUTH   -> URL path to auth folder (under ADMIN_BASE)
 * ADMINASSETS -> absolute URL to /assets under ADMIN_BASE
 */

// Path base used for validation and routing (leading & trailing slash)
if (!defined('ADMIN_BASE')) {
    define('ADMIN_BASE', '/admin-panel/'); // adjust if you deploy under a different path
}

// Absolute base URL (scheme + host + ADMIN_BASE)
if (!defined('ADMINURL')) {
    $https  = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off')
           || (($_SERVER['HTTP_X_FORWARDED_PROTO'] ?? '') === 'https');
    $scheme = $https ? 'https' : 'http';
    $host   = $_SERVER['HTTP_HOST'] ?? 'localhost';
    define('ADMINURL', $scheme . '://' . $host . ADMIN_BASE);
}

// Auth path (URL path, not full URL)
if (!defined('ADMINAUTH')) {
    define('ADMINAUTH', ADMIN_BASE . 'auth/');
}

// Assets absolute URL
if (!defined('ADMINASSETS')) {
    define('ADMINASSETS', ADMINURL . 'assets/');
}

/* -------------------- Helpers -------------------- */

// HTML escape
if (!function_exists('h')) {
    function h(?string $s): string {
        return htmlspecialchars((string)$s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
    }
}

/**
 * Only allow absolute paths inside ADMIN_BASE (blocks full URLs and outside paths).
 * Use this for validating a ?next= target before redirecting.
 */


/**
 * Build an absolute admin URL from a relative path under ADMIN_BASE.
 * Example: URL('dashboard.php') -> http(s)://host/Homeland/admin-panel/dashboard.php
 */
if (!function_exists('URL')) {
    function URL(string $path = ''): string {
        $path = ltrim($path, '/');
        return ADMINURL . $path;
    }
}

/**
 * Sanitize user input for safe echoing (basic).
 * For DB usage, also use prepared statements.
 */
if (!function_exists('sanitizeInput')) {
    function sanitizeInput(string $data): string {
        return htmlspecialchars(trim($data), ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
    }
}
if (!function_exists('str_starts_with')) {
    function str_starts_with(string $haystack, string $needle): bool {
        return $needle === '' || strpos($haystack, $needle) === 0;
    }
}

if (!function_exists('is_same_origin_path')) {
    function is_same_origin_path(string $path): bool {
        if ($path === '' || $path[0] !== '/') return false;
        return str_starts_with($path, ADMIN_BASE);
    }
}
