<?php
require_once "../includes/header.php";
require_once "../../functions/database.php";

try {
    $pdo = new PDO("mysql:host=localhost;dbname=homeland;charset=utf8mb4", "root", "");
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    die("DB Connection failed: " . $e->getMessage());
}

function sanitize($data) {
    return htmlspecialchars(trim($data));
}

$errors = [];
$success = '';
$propertyType = $_POST['property_type'] ?? '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit'])) {
    do {
        $code = str_pad(mt_rand(0, 9999), 4, '0', STR_PAD_LEFT);
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM properties WHERE code = ?");
        $stmt->execute([$code]);
    } while ($stmt->fetchColumn() > 0);

    $ownerName = sanitize($_POST['owner_name'] ?? '');
    $ownerPhone = sanitize($_POST['owner_phone'] ?? '');
    $title = sanitize($_POST['title'] ?? '');
    $description = sanitize($_POST['description'] ?? '');
    $adminId = $_SESSION['admin_id'] ?? null;

    if (!$ownerName) $errors['owner_name'] = "Owner name is required";
    if (!$ownerPhone) $errors['owner_phone'] = "Owner phone is required";
    if (!$propertyType) $errors['property_type'] = "Property type is required";
    if (!$title) $errors['title'] = "Property title is required.";

    $data = [
        'code' => $code,
        'owner_name' => $ownerName,
        'owner_phone' => $ownerPhone,
        'property_type' => $propertyType,
        'area' => sanitize($_POST['area'] ?? ''),
        'street_address' => sanitize($_POST['street'] ?? ''),
        'size_sqft' => (float)($_POST['size'] ?? 0),
        'directions' => sanitize($_POST['direction'] ?? ''),
        'unit' => sanitize($_POST['unit'] ?? ''),
        'year_built' => (int)($_POST['year'] ?? 0),
        'house_number' => sanitize($_POST['house_number'] ?? ''),
        'land_number' => sanitize($_POST['land_number'] ?? ''),
        'price' => (float)($_POST['price'] ?? 0),
        'note' => sanitize($_POST['note'] ?? ''),
        'residence_name' => sanitize($_POST['residence'] ?? ''),
        'floor' => (int)($_POST['floor'] ?? 0),
        'tower_number' => sanitize($_POST['tower'] ?? ''),
        'money_paid' => (float)($_POST['paid'] ?? 0),
        'monthly_installment' => (float)($_POST['monthly'] ?? 0),
        'remain_amount' => (float)($_POST['remaining'] ?? 0),
        'total_price' => (float)($_POST['total'] ?? 0),
        'title' => $title,
        'description' => $description,
        'admin_id' => $adminId,
        'rooms' => (int)($_POST['rooms'] ?? 0)
    ];

    if (!empty($_FILES['images']['name'][0])) {
        foreach ($_FILES['images']['tmp_name'] as $i => $tmpName) {
            $fileType = mime_content_type($tmpName);
            $fileSize = $_FILES['images']['size'][$i];
            if (!in_array($fileType, ['image/jpeg', 'image/png', 'image/gif'])) {
                $errors['images'] = "Only JPG, PNG, GIF allowed.";
                break;
            }
            if ($fileSize > 5 * 1024 * 1024) {
                $errors['images'] = "Each image must be less than 5MB.";
                break;
            }
        }
    } else {
        $errors['images'] = "Please upload at least one image.";
    }

    if (empty($errors)) {
        $stmt = $pdo->prepare("INSERT INTO properties (
            code, owner_name, owner_phone, property_type, area, street_address, size_sqft, directions, unit,
            year_built, house_number, land_number, price, note, residence_name, floor, tower_number,
            money_paid, monthly_installment, remain_amount, total_price, title, description, admin_id, rooms
        ) VALUES (
            :code, :owner_name, :owner_phone, :property_type, :area, :street_address, :size_sqft, :directions, :unit,
            :year_built, :house_number, :land_number, :price, :note, :residence_name, :floor, :tower_number,
            :money_paid, :monthly_installment, :remain_amount, :total_price, :title, :description, :admin_id, :rooms
        )");

        $stmt->execute($data);
        $propertyId = $pdo->lastInsertId();

        $uploadDir = __DIR__ . '/../assets/uploads/properties/galleries/';
        if (!is_dir($uploadDir)) mkdir($uploadDir, 0755, true);

        $stmtImg = $pdo->prepare("INSERT INTO galleries (property_id, image_url) VALUES (:property_id, :image_url)");
        foreach ($_FILES['images']['tmp_name'] as $i => $tmpName) {
            $fileName = uniqid('img_') . '.' . pathinfo($_FILES['images']['name'][$i], PATHINFO_EXTENSION);
            $dest = $uploadDir . $fileName;
            if (move_uploaded_file($tmpName, $dest)) {
                $stmtImg->execute([
                    ':property_id' => $propertyId,
                    ':image_url' => $fileName
                ]);
            }
        }

        $success = "Property added successfully with code: $code";
        $_POST = [];
        $propertyType = '';
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>زێدەکرنا مولکی</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet"/>
    <style>.form-section { display: none; }</style>
</head>
<body class="bg-light p-4">

<div class="container bg-white p-4 rounded shadow">
    <h2>زێدەکرنا مولکی</h2>

    <?php if ($success): ?>
        <div class="alert alert-success"><?= $success ?></div>
    <?php endif; ?>
    <?php if ($errors): ?>
        <div class="alert alert-danger">
            <ul><?php foreach ($errors as $err) echo "<li>$err</li>"; ?></ul>
        </div>
    <?php endif; ?>

    <form method="POST" enctype="multipart/form-data" novalidate>
        <div class="mb-3">
            <label>جورێ مولکی</label>
            <select name="property_type" id="propertyType" class="form-select" required onchange="toggleFields()">
                <option value="">-- Select --</option>
                <option value="house" <?= ($propertyType === 'house') ? 'selected' : '' ?>>خانی</option>
                <option value="land" <?= ($propertyType === 'land') ? 'selected' : '' ?>>عەرد</option>
                <option value="installment" <?= ($propertyType === 'installment') ? 'selected' : '' ?>>شقە قست</option>
                <option value="cash" <?= ($propertyType === 'cash') ? 'selected' : '' ?>>شقە کاش</option>
            </select>
        </div>

        <!-- Shared Fields -->
        <div class="mb-3"><label>ناڤێ خودانی</label><input type="text" name="owner_name" class="form-control" value="<?= htmlspecialchars($_POST['owner_name'] ?? '') ?>" required></div>
        <div class="mb-3"><label>ژمارا خودانی</label><input type="text" name="owner_phone" class="form-control" value="<?= htmlspecialchars($_POST['owner_phone'] ?? '') ?>" required></div>
        <div class="mb-3"><label>ناڤێ مولکی</label><input type="text" name="title" class="form-control" value="<?= htmlspecialchars($_POST['title'] ?? '') ?>" required></div>
        <div class="mb-3"><label>کورتی</label><textarea name="description" class="form-control" rows="3"><?= htmlspecialchars($_POST['description'] ?? '') ?></textarea></div>

        <!-- Dynamic Fields -->
        <div class="form-section" id="houseFields">
            <h5>House Details</h5>
            <input type="text" name="area" class="form-control mb-2" placeholder="گەرەگ" value="<?= htmlspecialchars($_POST['area'] ?? '') ?>">
            <input type="text" name="street" class="form-control mb-2" placeholder="جادە" value="<?= htmlspecialchars($_POST['street'] ?? '') ?>">
            <input type="number" step="0.01" name="size" class="form-control mb-2" placeholder="رووبەر" value="<?= htmlspecialchars($_POST['size'] ?? '') ?>">
            <input type="text" name="direction" class="form-control mb-2" placeholder="اتجاهـ" value="<?= htmlspecialchars($_POST['direction'] ?? '') ?>">
            <input type="text" name="unit" class="form-control mb-2" placeholder="تابق" value="<?= htmlspecialchars($_POST['unit'] ?? '') ?>">
            <input type="number" step="2010" name="year" class="form-control mb-2" placeholder="سالا ئاڤاکرنێ" value="<?= htmlspecialchars($_POST['year'] ?? '') ?>">
            <input type="text" name="house_number" class="form-control mb-2" placeholder="ژمارا خانی" value="<?= htmlspecialchars($_POST['house_number'] ?? '') ?>">
            <input type="number" name="rooms" class="form-control mb-2" placeholder="ژوور" value="<?= htmlspecialchars($_POST['rooms'] ?? '') ?>">
            <input type="number" step="0.01" name="price" class="form-control mb-2" placeholder="بها" value="<?= htmlspecialchars($_POST['price'] ?? '') ?>">
            <textarea name="note" class="form-control mb-2" placeholder="تێبینی"><?= htmlspecialchars($_POST['note'] ?? '') ?></textarea>
        </div>

        <div class="form-section" id="landFields">
            <h5>Land Details</h5>
            <input type="text" name="area" class="form-control mb-2" placeholder="گەرەک">
            <input type="text" name="street" class="form-control mb-2" placeholder="جادە">
            <input type="number" step="0.01" name="size" class="form-control mb-2" placeholder="رووبەر">
            <input type="text" name="direction" class="form-control mb-2" placeholder="اتجاهـ">
            <input type="text" name="land_number" class="form-control mb-2" placeholder="ژمارا عەردی">
            <input type="number" step="0.01" name="price" class="form-control mb-2" placeholder="بها">
        </div>

        <div class="form-section" id="installmentFields">
            <h5>شقە قست</h5>
            <input type="text" name="residence" class="form-control mb-2" placeholder="پروژە">
            <input type="number" step="0.01" name="size" class="form-control mb-2" placeholder="رووبەر">
            <input type="number" name="floor" class="form-control mb-2" placeholder="تابق">
            <input type="text" name="tower" class="form-control mb-2" placeholder="ژمارا تاوەری">
            <input type="text" name="direction" class="form-control mb-2" placeholder="اتجاهـ">
            <input type="number" name="rooms" class="form-control mb-2" placeholder="ژوور">
            <input type="number" step="0.01" name="paid" class="form-control mb-2" placeholder="پارێ هاتیە دان">
            <input type="number" step="0.01" name="monthly" class="form-control mb-2" placeholder="قستێ هەیڤانە">
            <input type="number" step="0.01" name="remaining" class="form-control mb-2" placeholder="پارێ مای">
            <input type="number" step="0.01" name="total" class="form-control mb-2" placeholder="مجموع پارێ شقێ">
        </div>

        <div class="form-section" id="cashFields">
            <h5>شقە کاش</h5>
            <input type="text" name="residence" class="form-control mb-2" placeholder="پروژە">
            <input type="number" step="0.01" name="size" class="form-control mb-2" placeholder="رووبەر">
            <input type="text" name="tower" class="form-control mb-2" placeholder="ژمارا تاوەری">
            <input type="text" name="direction" class="form-control mb-2" placeholder="اتجاهـ">
            <input type="number" name="rooms" class="form-control mb-2" placeholder="ژوور">
            <input type="number" step="0.01" name="price" class="form-control mb-2" placeholder="بها">
        </div>

        <div class="mb-3">
            <label>وێنە</label>
            <input type="file" name="images[]" class="form-control" multiple accept="image/*">
        </div>

        <button type="submit" name="submit" class="btn btn-primary">زێدە بکە</button>
    </form>
</div>

<script>
function toggleFields() {
  const type = document.getElementById('propertyType').value;

  // hide & disable everything first
  document.querySelectorAll('.form-section').forEach(section => {
    section.style.display = 'none';
    section.querySelectorAll('input, select, textarea').forEach(el => el.disabled = true);
  });

  // show & enable only the active section
  let activeId = null;
  if (type === 'house') activeId = 'houseFields';
  else if (type === 'land') activeId = 'landFields';
  else if (type === 'installment') activeId = 'installmentFields';
  else if (type === 'cash') activeId = 'cashFields';

  if (activeId) {
    const active = document.getElementById(activeId);
    active.style.display = 'block';
    active.querySelectorAll('input, select, textarea').forEach(el => el.disabled = false);
  }
}
window.onload = toggleFields;
</script>


</body>
</html>
