<?php require_once "../includes/header.php"; ?>
<?php require_once "../../functions/database.php"; ?>

<?php
try {
    $pdo = new PDO("mysql:host=localhost;dbname=homeland;charset=utf8mb4", "root", "");
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) { die("DB Connection failed: " . $e->getMessage()); }

/* -------------------- Helpers -------------------- */
$GET = fn($k) => isset($_GET[$k]) ? trim((string)$_GET[$k]) : '';
$HAS = fn($k) => array_key_exists($k, $_GET) && $_GET[$k] !== ''; // don't treat "0" as empty

function distinctCol(PDO $pdo, string $col): array {
    $stmt = $pdo->query("SELECT DISTINCT $col AS v FROM properties WHERE COALESCE($col,'')<>'' ORDER BY v");
    return $stmt->fetchAll(PDO::FETCH_COLUMN) ?: [];
}

/* -------------------- Inputs -------------------- */
$propertyType   = $GET('property_type'); // DB enum: House, Land, Apartment (Installment), Apartment (Cash)
$where   = [];
$params  = [];

/* Select options that come from DB */
$stockStatuses = distinctCol($pdo, 'status');

/* -------------------- WHERE builder -------------------- */
if ($propertyType !== '') {
    $where[] = "property_type = :property_type";
    $params[':property_type'] = $propertyType;

    if ($propertyType === 'House' || $propertyType === 'Land') {
        if ($HAS('area'))         { $where[] = "area LIKE :area";                         $params[':area'] = "%".$GET('area')."%"; }
        if ($HAS('street'))       { $where[] = "street_address LIKE :street_address";     $params[':street_address'] = "%".$GET('street')."%"; }
        if ($HAS('size'))         { $where[] = "size_sqft >= :size_min";                  $params[':size_min'] = (float)$GET('size'); }
        if ($HAS('size_max'))     { $where[] = "size_sqft <= :size_max";                  $params[':size_max'] = (float)$GET('size_max'); }
        if ($HAS('land_number'))  { $where[] = "land_number LIKE :land_number";           $params[':land_number'] = "%".$GET('land_number')."%"; }
        if ($HAS('house_number')) { $where[] = "house_number LIKE :house_number";         $params[':house_number'] = "%".$GET('house_number')."%"; }
        if ($HAS('unit'))         { $where[] = "unit LIKE :unit";                         $params[':unit'] = "%".$GET('unit')."%"; }
        if ($HAS('directions'))   { $where[] = "directions LIKE :directions";             $params[':directions'] = "%".$GET('directions')."%"; }
        if ($HAS('rooms'))        { $where[] = "rooms LIKE :rooms";                       $params[':rooms'] = "%".$GET('rooms')."%"; }
    }
    elseif ($propertyType === 'Apartment (Installment)' || $propertyType === 'Apartment (Cash)') {
        if ($HAS('residence'))    { $where[] = "residence_name LIKE :residence_name";     $params[':residence_name'] = "%".$GET('residence')."%"; }
        if ($HAS('size'))         { $where[] = "size_sqft >= :size_min";                  $params[':size_min'] = (float)$GET('size'); }
        if ($HAS('size_max'))     { $where[] = "size_sqft <= :size_max";                  $params[':size_max'] = (float)$GET('size_max'); }
        if ($HAS('floor'))        { $where[] = "floor = :floor";                          $params[':floor'] = $GET('floor'); }
        if ($HAS('tower_number')) { $where[] = "tower_number LIKE :tower_number";         $params[':tower_number'] = "%".$GET('tower_number')."%"; }
        if ($HAS('directions'))   { $where[] = "directions LIKE :directions";             $params[':directions'] = "%".$GET('directions')."%"; }
        if ($HAS('rooms'))        { $where[] = "rooms LIKE :rooms";                       $params[':rooms'] = "%".$GET('rooms')."%"; }
    }

    // Common price range
    if ($HAS('min_price') && is_numeric($GET('min_price'))) {
        $where[] = "price >= :min_price"; $params[':min_price'] = (float)$GET('min_price');
    }
    if ($HAS('max_price') && is_numeric($GET('max_price'))) {
        $where[] = "price <= :max_price"; $params[':max_price'] = (float)$GET('max_price');
    }

    // Status (كردي: بەردەستە/فروتیە)
    if ($HAS('status')) { $where[] = "status = :status"; $params[':status'] = $GET('status'); }

    // Year range (YEAR column)
    if ($HAS('year_min') && ctype_digit($GET('year_min'))) { $where[] = "year_built >= :year_min"; $params[':year_min'] = (int)$GET('year_min'); }
    if ($HAS('year_max') && ctype_digit($GET('year_max'))) { $where[] = "year_built <= :year_max"; $params[':year_max'] = (int)$GET('year_max'); }
}

/* -------------------- Query -------------------- */
$sql = "SELECT * FROM properties";
if ($where) $sql .= " WHERE ".implode(" AND ", $where);
$sql .= " ORDER BY created_at DESC";

$stmt = $pdo->prepare($sql);
foreach ($params as $k=>$v) { $stmt->bindValue($k, $v); }
$stmt->execute();
$results = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<div class="container mt-5 pt-4">
  <h2>Search Properties</h2>

  <!-- Add id for JS auto-search -->
  <form id="searchForm" class="row g-3 mb-4 bg-light p-3 rounded" method="GET">
      <div class="d-flex justify-content-between align-items-center mb-3">
    <h4 class="mb-0">فلتەرەکان</h4>
    <!-- Reset button -->
    <a href="<?= basename($_SERVER['PHP_SELF']) ?>" class="btn btn-outline-secondary btn-sm">
      🗑️ ریست/نوی کرنا فورمی
    </a>
  </div>
    <div class="col-md-3">
      <label class="form-label">جورێ مولکی</label>
      <select name="property_type" class="form-select" onchange="this.form.submit()">
        <option value="">-- All --</option>
        <option value="House" <?= ($propertyType === 'House') ? 'selected' : '' ?>>خانی</option>
        <option value="Land" <?= ($propertyType === 'Land') ? 'selected' : '' ?>>عەرد</option>
        <option value="Apartment (Installment)" <?= ($propertyType === 'Apartment (Installment)') ? 'selected' : '' ?>>شقە قست</option>
        <option value="Apartment (Cash)" <?= ($propertyType === 'Apartment (Cash)') ? 'selected' : '' ?>>شقە کاش</option>
      </select>
    </div>

    <div class="col-md-2">
      <label class="form-label">دۆخ</label>
      <select name="status" class="form-select">
        <option value="">-- Any --</option>
        <?php foreach ($stockStatuses as $ss): ?>
          <option value="<?= htmlspecialchars($ss) ?>" <?= (($_GET['status'] ?? '') === $ss) ? 'selected' : '' ?>>
            <?= htmlspecialchars($ss) ?>
          </option>
        <?php endforeach; ?>
      </select>
    </div>
    <div class="col-md-2">
      <label class="form-label">ساڵ ژ</label>
      <input type="number" name="year_min" min="1900" max="2099" class="form-control" value="<?= htmlspecialchars($_GET['year_min'] ?? '') ?>">
    </div>
    <div class="col-md-2">
      <label class="form-label">ساڵ هەتا</label>
      <input type="number" name="year_max" min="1900" max="2099" class="form-control" value="<?= htmlspecialchars($_GET['year_max'] ?? '') ?>">
    </div>

    <?php if ($propertyType === 'House' || $propertyType === 'Land'): ?>
      <div class="col-md-3">
        <label>گەرەک</label>
        <input type="text" name="area" class="form-control" value="<?= htmlspecialchars($_GET['area'] ?? '') ?>">
      </div>
      <div class="col-md-3">
        <label>جادە</label>
        <input type="text" name="street" class="form-control" value="<?= htmlspecialchars($_GET['street'] ?? '') ?>">
      </div>
      <div class="col-md-2">
        <label>رووبەر ژ-</label>
        <input type="number" step="0.01" name="size" class="form-control" value="<?= htmlspecialchars($_GET['size'] ?? '') ?>">
      </div>
      <div class="col-md-2">
        <label>رووبەر هەتا</label>
        <input type="number" step="0.01" name="size_max" class="form-control" value="<?= htmlspecialchars($_GET['size_max'] ?? '') ?>">
      </div>
      <div class="col-md-2">
        <label>رێنمە</label>
        <input type="text" name="directions" class="form-control" value="<?= htmlspecialchars($_GET['directions'] ?? '') ?>">
      </div>
      <div class="col-md-2">
        <label>ژمارا ژوور</label>
        <input type="text" name="rooms" class="form-control" value="<?= htmlspecialchars($_GET['rooms'] ?? '') ?>">
      </div>
      <div class="col-md-2">
        <label>تابق</label>
        <input type="text" name="unit" class="form-control" value="<?= htmlspecialchars($_GET['unit'] ?? '') ?>">
      </div>
      <div class="col-md-2">
        <label>ژمارا عەردێ</label>
        <input type="text" name="land_number" class="form-control" value="<?= htmlspecialchars($_GET['land_number'] ?? '') ?>">
      </div>
      <div class="col-md-2">
        <label>ژمارا خانی</label>
        <input type="text" name="house_number" class="form-control" value="<?= htmlspecialchars($_GET['house_number'] ?? '') ?>">
      </div>
      <div class="col-md-2">
        <label>بها ژ</label>
        <input type="number" step="0.01" name="min_price" class="form-control" value="<?= htmlspecialchars($_GET['min_price'] ?? '') ?>">
      </div>
      <div class="col-md-2">
        <label>بها هەتا</label>
        <input type="number" step="0.01" name="max_price" class="form-control" value="<?= htmlspecialchars($_GET['max_price'] ?? '') ?>">
      </div>

    <?php elseif ($propertyType === 'Apartment (Installment)' || $propertyType === 'Apartment (Cash)'): ?>
      <div class="col-md-4">
        <label>پروژە / رزیدانس</label>
        <input type="text" name="residence" class="form-control" value="<?= htmlspecialchars($_GET['residence'] ?? '') ?>">
      </div>
      <div class="col-md-2">
        <label>رووبەر ژ-</label>
        <input type="number" step="0.01" name="size" class="form-control" value="<?= htmlspecialchars($_GET['size'] ?? '') ?>">
      </div>
      <div class="col-md-2">
        <label>رووبەر هەتا</label>
        <input type="number" step="0.01" name="size_max" class="form-control" value="<?= htmlspecialchars($_GET['size_max'] ?? '') ?>">
      </div>
      <div class="col-md-2">
        <label>تابق</label>
        <input type="text" name="floor" class="form-control" value="<?= htmlspecialchars($_GET['floor'] ?? '') ?>">
      </div>
      <div class="col-md-2">
        <label>ژمارا تاوەر</label>
        <input type="text" name="tower_number" class="form-control" value="<?= htmlspecialchars($_GET['tower_number'] ?? '') ?>">
      </div>
      <div class="col-md-2">
        <label>رێنمە</label>
        <input type="text" name="directions" class="form-control" value="<?= htmlspecialchars($_GET['directions'] ?? '') ?>">
      </div>
      <div class="col-md-2">
        <label>ژمارا ژوور</label>
        <input type="text" name="rooms" class="form-control" value="<?= htmlspecialchars($_GET['rooms'] ?? '') ?>">
      </div>
      <div class="col-md-2">
        <label>بها ژ</label>
        <input type="number" step="0.01" name="min_price" class="form-control" value="<?= htmlspecialchars($_GET['min_price'] ?? '') ?>">
      </div>
      <div class="col-md-2">
        <label>بها هەتا</label>
        <input type="number" step="0.01" name="max_price" class="form-control" value="<?= htmlspecialchars($_GET['max_price'] ?? '') ?>">
      </div>
    <?php endif; ?>

    <div class="col-md-2 align-self-end">
      <button class="btn btn-primary w-100">لێگەریان</button>
    </div>
  </form>

  <div class="row">
    <?php if (!$results): ?>
      <div class="col-12"><div class="alert alert-warning">نە هاتە دیتن</div></div>
    <?php else: ?>
      <?php foreach ($results as $prop): ?>
        <?php
          $imgStmt = $pdo->prepare("SELECT image_url FROM galleries WHERE property_id = ? LIMIT 1");
          $imgStmt->execute([$prop['id']]);
          $image = $imgStmt->fetchColumn();
          $imageUrl = $image ? "../assets/uploads/properties/galleries/$image" : "assets/img/no-image.jpg";
        ?>
        <div class="col-md-6 col-lg-4 mb-4">
          <div class="card h-100">
            <img src="<?= htmlspecialchars($imageUrl) ?>" class="card-img-top" alt="Image">
            <div class="card-body">
              <h5 class="card-title"><?= htmlspecialchars($prop['title']) ?></h5>
              <p class="mb-1"><strong>جور:</strong> <?= htmlspecialchars($prop['property_type']) ?></p>
              <p class="mb-1"><strong>گەڕەک:</strong> <?= htmlspecialchars($prop['area'] ?? '—') ?></p>
              <p class="mb-1"><strong>رووبەر:</strong> <?= htmlspecialchars($prop['size_sqft']) ?> m²</p>
              <p class="mb-1"><strong>بها:</strong> $<?= number_format((float)$prop['price']) ?></p>
              <a href="../property-details.php?id=<?= (int)$prop['id'] ?>" class="btn btn-outline-primary btn-sm mt-2">View</a>
            </div>
          </div>
        </div>
      <?php endforeach; ?>
    <?php endif; ?>
  </div>
</div>

<?php require_once "../includes/footer.php"; ?>
<script>
/* Auto-search: 
   - selects submit instantly on change
   - text/number/search submit ONLY on Enter or when the field loses focus (no typing submit)
*/
(function () {
  const form = document.getElementById('searchForm');
  if (!form) return;

  const submitNow = () => form.submit();

  // Selects -> submit on change (property_type already has inline onchange)
  form.querySelectorAll('select').forEach(sel => {
    if (sel.name !== 'property_type') {
      sel.addEventListener('change', submitNow);
    }
  });

  // Text-like inputs: Enter or blur/change only (no 'input' listener)
  const TEXTY = ['text', 'number', 'search', 'tel', 'email'];
  form.querySelectorAll('input').forEach(inp => {
    const t = (inp.type || '').toLowerCase();
    if (!TEXTY.includes(t)) return;

    // Pressing Enter submits immediately
    inp.addEventListener('keydown', (e) => {
      if (e.key === 'Enter') {
        e.preventDefault();
        submitNow();
      }
    });

    // When user leaves the field (and value changed), submit
    inp.addEventListener('change', submitNow);
    // Optional extra safety: submit on blur even if browser doesn't fire change
    // inp.addEventListener('blur', submitNow);
  });
})();
</script>
