<?php require_once "../includes/header.php"; ?>
<?php require_once "../../functions/database.php"; ?>
<?php
if (session_status() === PHP_SESSION_NONE) session_start();

/* ---------------- Helpers ---------------- */
function h($v){ return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }
function v($key,$def=''){ return $_POST[$key] ?? $def; }
function vDetail($key,$def=''){ return $_POST['details'][$key] ?? $def; }
function toNull($v){
  if ($v === null) return null;
  $v = trim((string)$v);
  return ($v === '') ? null : $v;
}
function toMoney($v){
  $v = toNull($v);
  if ($v === null) return null;
  // allow 1,234.56 or 1234.56
  $norm = preg_replace('/[,\s]/','',$v);
  return is_numeric($norm) ? $norm : null;
}
function num($v, $allowFloat = true){
  $v = toNull($v);
  if ($v === null) return null;
  $norm = preg_replace('/[,\s]/','',$v);
  if (!is_numeric($norm)) return null;
  return $allowFloat ? (float)$norm : (int)$norm;
}
function ensureCsrf(){
  if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
  return $_SESSION['csrf'];
}
function checkCsrf(){
  return isset($_POST['csrf']) && hash_equals($_SESSION['csrf'] ?? '', $_POST['csrf']);
}

/* ---------------- PDO safety ---------------- */
if (isset($pdo) && $pdo instanceof PDO) {
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
}

/* ---------------- Init ---------------- */
$success = '';
$errors  = [];

/* ---------------- Handle POST ---------------- */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

  if (!checkCsrf()) {
    $errors['csrf'] = "Security check failed. Please try again.";
  }

  // Property type: map legacy → canonical + whitelist
  $type = v('property_type','');
  $legacyMap = [
    'installment_apartment' => 'apartment_installment',
    'cash_apartment'        => 'apartment_cash',
  ];
  if (isset($legacyMap[$type])) $type = $legacyMap[$type];

  $allowedTypes = ['house','land','apartment_installment','apartment_cash'];
  if (!in_array($type, $allowedTypes, true)) $type = '';

  // Basics
  $clientName  = trim(v('client_name',''));
  $clientPhone = trim(v('client_phone',''));

  if ($clientName === '')  $errors['client_name'] = "Client name is required.";
  if ($clientPhone === '') $errors['client_phone'] = "Client phone is required.";
  if ($type === '')        $errors['property_type'] = "Select a valid property type.";

  // Extract shared / details fields
  $price  = v('price', vDetail('price'));                              // shared top-level or per-type
  $note   = v('note');

  $area   = v('area',   vDetail('area'));
  $street = v('street', vDetail('street'));
  $size   = v('size_meters', vDetail('size'));

  // directions can come as details[directions] or details[direction]
  $directions = v('directions', vDetail('directions', vDetail('direction')));

  $unit           = vDetail('unit');
  $builded_year   = vDetail('builded_year');
  $land_number    = vDetail('land_number');
  $residence_name = vDetail('residence_name');
  $floor          = vDetail('floor');
  $tower_number   = vDetail('tower_number');
  $monthly_inst   = vDetail('monthly_installment');

  // ---- Validation rules ----
  // phone (light check)
  if ($clientPhone !== '' && !preg_match('/^[0-9+\-\s()]{6,20}$/', $clientPhone)) {
    $errors['client_phone'] = "Enter a valid phone number.";
  }

  // numeric fields
  $sizeNum  = num($size, true);
  if ($size !== null && $size !== '' && $sizeNum === null) {
    $errors['size'] = "Size must be a number.";
  }

  $yearInt = num($builded_year, false);
  if ($builded_year !== null && $builded_year !== '' ){
    $y = (int)$yearInt;
    $curY = (int)date('Y');
    if ($yearInt === null || $y < 1900 || $y > ($curY+1)) {
      $errors['builded_year'] = "Build year must be between 1900 and ".($curY+1).".";
    }
  }

  $priceNum = toMoney($price);
  if ($price !== null && $price !== '' && $priceNum === null) {
    $errors['price'] = "Price must be numeric.";
  }

  $monthlyNum = toMoney($monthly_inst);
  if ($monthly_inst !== null && $monthly_inst !== '' && $monthlyNum === null) {
    $errors['monthly_installment'] = "Monthly installment must be numeric.";
  }

  // required per type (tweak as you prefer)
  if ($type === 'house') {
    if (toNull($area) === null)        $errors['house.area']   = "Area is required for houses.";
    if (toNull($directions) === null)  $errors['house.dir']    = "Directions are required for houses.";
  } elseif ($type === 'land') {
    if (toNull($size) === null)        $errors['land.size']    = "Size is required for land.";
    if (toNull($directions) === null)  $errors['land.dir']     = "Directions are required for land.";
  } elseif ($type === 'apartment_installment') {
    if (toNull($residence_name) === null) $errors['ai.res']    = "Residence name is required.";
    if (toNull($monthly_inst) === null)   $errors['ai.month']  = "Monthly installment is required.";
  } elseif ($type === 'apartment_cash') {
    if (toNull($residence_name) === null) $errors['ac.res']    = "Residence name is required.";
    if (toNull($price) === null)          $errors['ac.price']  = "Price is required.";
  }

  // ---- Insert ----
  if (empty($errors)) {
    try {
      $stmt = $pdo->prepare("
        INSERT INTO client_requests
          (client_name, client_phone, property_type, area, street, size_meters, directions, price,
           note, unit, builded_year, land_number, residence_name, floor, tower_number, monthly_installment, created_at)
        VALUES
          (:client_name, :client_phone, :property_type, :area, :street, :size_meters, :directions, :price,
           :note, :unit, :builded_year, :land_number, :residence_name, :floor, :tower_number, :monthly_installment, NOW())
      ");

      $stmt->execute([
        ':client_name'         => $clientName,
        ':client_phone'        => $clientPhone,
        ':property_type'       => $type,
        ':area'                => toNull($area),
        ':street'              => toNull($street),
        ':size_meters'         => $sizeNum,
        ':directions'          => toNull($directions),
        ':price'               => $priceNum,
        ':note'                => toNull($note),
        ':unit'                => toNull($unit),
        ':builded_year'        => $yearInt,
        ':land_number'         => toNull($land_number),
        ':residence_name'      => toNull($residence_name),
        ':floor'               => num($floor, false),
        ':tower_number'        => toNull($tower_number),
        ':monthly_installment' => $monthlyNum,
      ]);

      $success = "Client request submitted successfully.";
      $_POST = []; // clear form
      // rotate CSRF to avoid re-post
      $_SESSION['csrf'] = bin2hex(random_bytes(16));
    } catch (PDOException $e) {
      $errors['db'] = "Database error: ".$e->getMessage();
    }
  }
}

$csrf = ensureCsrf();
$selectedType = v('property_type','');
?>
<div class="container mt-5 pt-5">
  <h2 class="mb-3">Add Client Request</h2>

  <?php if (!empty($errors['db'])): ?>
    <div class="alert alert-danger"><?= h($errors['db']) ?></div>
  <?php endif; ?>
  <?php if (!empty($errors['csrf'])): ?>
    <div class="alert alert-danger"><?= h($errors['csrf']) ?></div>
  <?php endif; ?>
  <?php if ($success): ?>
    <div class="alert alert-success"><?= h($success) ?></div>
  <?php endif; ?>

  <form method="POST" class="mt-3" novalidate>
    <input type="hidden" name="csrf" value="<?= h($csrf) ?>">

    <div class="row g-3">
      <div class="col-md-6">
        <label class="form-label">Client Name</label>
        <input type="text" name="client_name" class="form-control" value="<?= h(v('client_name')) ?>" required>
        <?php if (isset($errors['client_name'])): ?><div class="text-danger small"><?= h($errors['client_name']) ?></div><?php endif; ?>
      </div>

      <div class="col-md-6">
        <label class="form-label">Client Phone</label>
        <input type="text" name="client_phone" class="form-control" value="<?= h(v('client_phone')) ?>" required>
        <?php if (isset($errors['client_phone'])): ?><div class="text-danger small"><?= h($errors['client_phone']) ?></div><?php endif; ?>
      </div>

      <div class="col-md-6">
        <label class="form-label">Property Type</label>
        <select name="property_type" class="form-select" id="property-type" required>
          <option value="">-- Select Type --</option>
          <option value="house" <?= $selectedType==='house' ? 'selected' : '' ?>>House</option>
          <option value="land" <?= $selectedType==='land' ? 'selected' : '' ?>>Land</option>
          <option value="apartment_installment" <?= $selectedType==='apartment_installment' ? 'selected' : '' ?>>Apartment with Installment</option>
          <option value="apartment_cash" <?= $selectedType==='apartment_cash' ? 'selected' : '' ?>>Cash Apartment</option>
        </select>
        <?php if (isset($errors['property_type'])): ?><div class="text-danger small"><?= h($errors['property_type']) ?></div><?php endif; ?>
      </div>

      <div class="col-12">
        <label class="form-label">Note</label>
        <textarea name="note" class="form-control" rows="3"><?= h(v('note')) ?></textarea>
      </div>
    </div>

    <hr class="my-4">

    <!-- HOUSE -->
    <div class="type-section" id="type-house" style="display:none;">
      <h5 class="mb-3">House Details</h5>
      <div class="row g-3">
        <div class="col-md-4"><label class="form-label">Area</label><input type="text" name="details[area]" class="form-control" value="<?= h(vDetail('area')) ?>"></div>
        <div class="col-md-4"><label class="form-label">Street</label><input type="text" name="details[street]" class="form-control" value="<?= h(vDetail('street')) ?>"></div>
        <div class="col-md-4"><label class="form-label">Size (m²)</label><input type="text" name="details[size]" class="form-control" inputmode="decimal" value="<?= h(vDetail('size')) ?>"><?php if (isset($errors['size'])): ?><div class="text-danger small"><?= h($errors['size']) ?></div><?php endif; ?></div>
        <div class="col-md-4"><label class="form-label">Directions</label><input type="text" name="details[directions]" class="form-control" value="<?= h(vDetail('directions')) ?>"></div>
        <div class="col-md-4"><label class="form-label">Unit</label><input type="text" name="details[unit]" class="form-control" value="<?= h(vDetail('unit')) ?>"></div>
        <div class="col-md-4"><label class="form-label">Build Year</label><input type="text" name="details[builded_year]" class="form-control" inputmode="numeric" value="<?= h(vDetail('builded_year')) ?>"><?php if (isset($errors['builded_year'])): ?><div class="text-danger small"><?= h($errors['builded_year']) ?></div><?php endif; ?></div>
        <div class="col-md-4"><label class="form-label">Price</label><input type="text" name="details[price]" class="form-control" inputmode="decimal" value="<?= h(vDetail('price')) ?>"></div>
        <?php if (isset($errors['house.area'])): ?><div class="col-12 text-danger small"><?= h($errors['house.area']) ?></div><?php endif; ?>
        <?php if (isset($errors['house.dir'])): ?><div class="col-12 text-danger small"><?= h($errors['house.dir']) ?></div><?php endif; ?>
      </div>
    </div>

    <!-- LAND -->
    <div class="type-section" id="type-land" style="display:none;">
      <h5 class="mb-3">Land Details</h5>
      <div class="row g-3">
        <div class="col-md-4"><label class="form-label">Area</label><input type="text" name="details[area]" class="form-control" value="<?= h(vDetail('area')) ?>"></div>
        <div class="col-md-4"><label class="form-label">Street</label><input type="text" name="details[street]" class="form-control" value="<?= h(vDetail('street')) ?>"></div>
        <div class="col-md-4"><label class="form-label">Size (m²)</label><input type="text" name="details[size]" class="form-control" inputmode="decimal" value="<?= h(vDetail('size')) ?>"></div>
        <div class="col-md-4"><label class="form-label">Directions</label><input type="text" name="details[directions]" class="form-control" value="<?= h(vDetail('directions')) ?>"></div>
        <div class="col-md-4"><label class="form-label">Land Number</label><input type="text" name="details[land_number]" class="form-control" value="<?= h(vDetail('land_number')) ?>"></div>
        <div class="col-md-4"><label class="form-label">Price</label><input type="text" name="details[price]" class="form-control" inputmode="decimal" value="<?= h(vDetail('price')) ?>"></div>
        <?php if (isset($errors['land.size'])): ?><div class="col-12 text-danger small"><?= h($errors['land.size']) ?></div><?php endif; ?>
        <?php if (isset($errors['land.dir'])): ?><div class="col-12 text-danger small"><?= h($errors['land.dir']) ?></div><?php endif; ?>
      </div>
    </div>

    <!-- APARTMENT WITH INSTALLMENT -->
    <div class="type-section" id="type-apartment_installment" style="display:none;">
      <h5 class="mb-3">Installment Apartment Details</h5>
      <div class="row g-3">
        <div class="col-md-4"><label class="form-label">Residence Name</label><input type="text" name="details[residence_name]" class="form-control" value="<?= h(vDetail('residence_name')) ?>"></div>
        <div class="col-md-4"><label class="form-label">Size (m²)</label><input type="text" name="details[size]" class="form-control" inputmode="decimal" value="<?= h(vDetail('size')) ?>"></div>
        <div class="col-md-4"><label class="form-label">Floor</label><input type="text" name="details[floor]" class="form-control" inputmode="numeric" value="<?= h(vDetail('floor')) ?>"></div>
        <div class="col-md-4"><label class="form-label">Tower Number</label><input type="text" name="details[tower_number]" class="form-control" value="<?= h(vDetail('tower_number')) ?>"></div>
        <div class="col-md-4"><label class="form-label">Direction</label><input type="text" name="details[direction]" class="form-control" value="<?= h(vDetail('direction')) ?>"></div>
        <div class="col-md-4"><label class="form-label">Monthly Installment</label><input type="text" name="details[monthly_installment]" class="form-control" inputmode="decimal" value="<?= h(vDetail('monthly_installment')) ?>"></div>
        <?php if (isset($errors['ai.res'])): ?><div class="col-12 text-danger small"><?= h($errors['ai.res']) ?></div><?php endif; ?>
        <?php if (isset($errors['ai.month'])): ?><div class="col-12 text-danger small"><?= h($errors['ai.month']) ?></div><?php endif; ?>
      </div>
    </div>

    <!-- CASH APARTMENT -->
    <div class="type-section" id="type-apartment_cash" style="display:none;">
      <h5 class="mb-3">Cash Apartment Details</h5>
      <div class="row g-3">
        <div class="col-md-4"><label class="form-label">Residence Name</label><input type="text" name="details[residence_name]" class="form-control" value="<?= h(vDetail('residence_name')) ?>"></div>
        <div class="col-md-4"><label class="form-label">Size (m²)</label><input type="text" name="details[size]" class="form-control" inputmode="decimal" value="<?= h(vDetail('size')) ?>"></div>
        <div class="col-md-4"><label class="form-label">Direction</label><input type="text" name="details[direction]" class="form-control" value="<?= h(vDetail('direction')) ?>"></div>
        <div class="col-md-4"><label class="form-label">Price</label><input type="text" name="details[price]" class="form-control" inputmode="decimal" value="<?= h(vDetail('price')) ?>"></div>
        <?php if (isset($errors['ac.res'])): ?><div class="col-12 text-danger small"><?= h($errors['ac.res']) ?></div><?php endif; ?>
        <?php if (isset($errors['ac.price'])): ?><div class="col-12 text-danger small"><?= h($errors['ac.price']) ?></div><?php endif; ?>
      </div>
    </div>

    <button type="submit" class="btn btn-primary mt-4">Submit Request</button>
  </form>
</div>

<script>
document.addEventListener('DOMContentLoaded', function () {
  const typeSelect = document.getElementById('property-type');
  const sections   = document.querySelectorAll('.type-section');

  function showSection(type) {
    sections.forEach(section => {
      section.style.display = 'none';
      section.querySelectorAll('input,select,textarea').forEach(el => el.disabled = true);
    });
    if (!type) return;
    const active = document.getElementById('type-' + type);
    if (active) {
      active.style.display = 'block';
      active.querySelectorAll('input,select,textarea').forEach(el => el.disabled = false);
    }
  }

  typeSelect.addEventListener('change', function(){ showSection(this.value); });
  showSection(typeSelect.value); // on load / postback
});
</script>

<?php require_once "../includes/footer.php"; ?>
