<?php
// requests/get_client_requests.php
declare(strict_types=1);
header('Content-Type: application/json; charset=utf-8');

// ---- harden & debug: capture stray output so we can still return valid JSON
ob_start();
ini_set('display_errors', '0');
error_reporting(E_ALL);

use PDO;
use PDOException;

$tz = 'Asia/Baghdad';
date_default_timezone_set($tz);

// ---------- helpers ----------
function out_json(array $payload, int $code = 200): void {
  http_response_code($code);
  // If there was stray output, attach it (helps diagnose)
  $leak = trim(ob_get_clean() ?: '');
  if ($leak !== '') $payload['_leak'] = $leak;
  echo json_encode($payload, JSON_UNESCAPED_UNICODE);
  exit;
}
function fail(string $msg, int $code = 500, array $ex = []): void {
  out_json(['ok' => false, 'error' => $msg, 'extra' => $ex], $code);
}

// ---------- DB ----------
$pdo = $pdo ?? null;

// Try common includes to reuse $pdo if your app already defines it
foreach ([__DIR__.'/../includes/header.php', __DIR__.'/../../includes/header.php', __DIR__.'/../../functions/database.php'] as $inc) {
  if ($pdo instanceof PDO) break;
  if (is_file($inc)) require_once $inc;
}

// Fallback DSN (edit to your env if needed)
if (!$pdo instanceof PDO) {
  try {
    $dbHost = getenv('DB_HOST') ?: '127.0.0.1';
    $dbName = getenv('DB_NAME') ?: 'homeland';
    $dbUser = getenv('DB_USER') ?: 'root';
    $dbPass = getenv('DB_PASS') ?: '';
    $pdo = new PDO("mysql:host=$dbHost;dbname=$dbName;charset=utf8mb4", $dbUser, $dbPass, [
      PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
      PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    ]);
  } catch (PDOException $e) {
    fail('Database connection failed', 500, ['pdo_message' => $e->getMessage()]);
  }
}

// ---------- validate schema (friendly messages) ----------
try {
  // table exists?
  $chk = $pdo->query("SHOW TABLES LIKE 'client_requests'")->fetch();
  if (!$chk) fail("Table 'client_requests' not found. Create it or adjust endpoint.", 500);

  // created_at exists?
  $cols = $pdo->query("SHOW COLUMNS FROM client_requests")->fetchAll();
  $colNames = array_map(fn($r) => $r['Field'], $cols);
  if (!in_array('created_at', $colNames, true)) {
    fail("Column 'created_at' missing in 'client_requests'. Add DATETIME column.", 500, ['columns' => $colNames]);
  }

  if (!in_array('property_type', $colNames, true)) {
    // Not fatal; we’ll ignore filter if column missing
    $hasPropertyType = false;
  } else {
    $hasPropertyType = true;
  }
} catch (PDOException $e) {
  fail('Schema check failed', 500, ['pdo_message' => $e->getMessage()]);
}

// ---------- inputs ----------
$months = max(1, min((int)($_GET['months'] ?? 12), 36));
$propertyType = isset($_GET['property_type']) ? trim((string)$_GET['property_type']) : null;

// time window: whole months
$now = new DateTime('now', new DateTimeZone($tz));
$end = (clone $now)->modify('first day of next month')->setTime(0, 0, 0);
$start = (clone $end)->modify('-'.$months.' months');

// labels & month keys
$cursor = clone $start;
$labels = [];
$ymKeys = [];
while ($cursor < $end) {
  $ymKeys[] = $cursor->format('Y-m');
  $labels[] = $cursor->format('M Y');
  $cursor->modify('+1 month');
}

// ---------- query ----------
try {
  $sql = "
    SELECT DATE_FORMAT(created_at, '%Y-%m') AS ym, COUNT(*) AS cnt
    FROM client_requests
    WHERE created_at >= :start AND created_at < :end
  ";
  $params = [
    ':start' => $start->format('Y-m-d H:i:s'),
    ':end'   => $end->format('Y-m-d H:i:s'),
  ];

  if ($hasPropertyType && $propertyType !== null && $propertyType !== '') {
    $sql .= " AND property_type = :ptype";
    $params[':ptype'] = $propertyType;
  }

  $sql .= " GROUP BY ym ORDER BY ym ASC";
  $stmt = $pdo->prepare($sql);
  $stmt->execute($params);
  $rows = $stmt->fetchAll();

  $map = [];
  foreach ($rows as $r) $map[$r['ym']] = (int)$r['cnt'];

  $data = [];
  $total = 0;
  foreach ($ymKeys as $k) {
    $v = $map[$k] ?? 0;
    $data[] = $v;
    $total += $v;
  }

  out_json([
    'ok' => true,
    'labels' => $labels,
    'months' => $ymKeys,
    'data' => $data,
    'total' => $total,
    'meta' => [
      'range_months' => $months,
      'start' => $start->format('Y-m-d'),
      'end' => (clone $end)->modify('-1 second')->format('Y-m-d'),
      'generated_at' => $now->format('Y-m-d H:i:s'),
      'timezone' => $tz,
      'filters' => [
        // if property_type column is missing, we surface that for clarity
        'property_type' => $hasPropertyType ? $propertyType : null,
        'property_type_column_exists' => $hasPropertyType,
      ],
    ],
  ]);

} catch (PDOException $e) {
  fail('Query failed', 500, ['pdo_message' => $e->getMessage()]);
}
