<?php require_once "../includes/header.php"; ?>
<?php require_once "../../functions/database.php"; ?>

<?php
// ---------- Helpers ----------
function h($v) { return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }
function showVal($v, $fallback = '—') {
    if ($v === null) return $fallback;
    $v = trim((string)$v);
    return $v === '' ? $fallback : h($v);
}
function moneyVal($v) {
    if ($v === null || $v === '' || (float)$v == 0.0) return '—';
    return '$' . number_format((float)$v, 2);
}
function normTypeForProperties($t) {
    // client_requests: house, land, apartment_installment, apartment_cash
    // properties:      house, land, installment, cash
    if ($t === 'apartment_installment') return 'installment';
    if ($t === 'apartment_cash') return 'cash';
    return $t;
}

// Ensure PDO errors are visible if your database.php didn't set it
if (isset($pdo) && $pdo instanceof PDO) {
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
}

// ---------- Load Request ----------
$id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$reqStmt = $pdo->prepare("SELECT * FROM client_requests WHERE id = ?");
$reqStmt->execute([$id]);
$request = $reqStmt->fetch(PDO::FETCH_ASSOC);
?>

<div class="container mt-5 pt-5">

<?php if (!$request): ?>
    <div class="alert alert-warning">Request not found.</div>
    <a href="client-requests.php" class="btn btn-secondary mt-2">Back to Requests</a>
    <?php require_once "../includes/footer.php"; exit; ?>
<?php endif; ?>

    <div class="d-flex align-items-center justify-content-between mb-3">
        <h2 class="mb-0">Request #<?= (int)$request['id'] ?></h2>
        <a href="client-requests.php" class="btn btn-outline-secondary btn-sm">Back</a>
    </div>

    <!-- Request details (all columns) -->
    <div class="card shadow-sm mb-4">
        <div class="card-body">
            <div class="d-flex justify-content-between align-items-start mb-3">
                <div>
                    <h4 class="mb-1"><?= showVal($request['client_name']) ?></h4>
                    <div class="text-muted small">Submitted on <?= h(date('Y-m-d H:i', strtotime($request['created_at']))) ?></div>
                </div>
                <span class="badge bg-primary-subtle text-primary border border-primary-subtle" style="text-transform:capitalize;">
                    <?= showVal(str_replace('_', ' ', $request['property_type'])) ?>
                </span>
            </div>

            <div class="table-responsive">
                <table class="table table-sm align-middle mb-0">
                    <tbody>
                        <tr><th style="width:260px;">Client Name</th><td><?= showVal($request['client_name']) ?></td></tr>
                        <tr><th>Client Phone</th><td><?= showVal($request['client_phone']) ?></td></tr>
                        <tr><th>Property Type</th><td><?= showVal(str_replace('_', ' ', $request['property_type'])) ?></td></tr>
                        <tr><th>Area</th><td><?= showVal($request['area']) ?></td></tr>
                        <tr><th>Street</th><td><?= showVal($request['street']) ?></td></tr>
                        <tr><th>Size (m²)</th><td><?= showVal($request['size_meters']) ?></td></tr>
                        <tr><th>Directions</th><td><?= showVal($request['directions']) ?></td></tr>
                        <tr><th>Target Price / Budget</th><td><?= moneyVal($request['price']) ?></td></tr>
                        <tr><th>Note</th><td><?= nl2br(showVal($request['note'])) ?></td></tr>
                        <tr><th>Unit</th><td><?= showVal($request['unit']) ?></td></tr>
                        <tr><th>Builded Year</th><td><?= showVal($request['builded_year']) ?></td></tr>
                        <tr><th>Land Number</th><td><?= showVal($request['land_number']) ?></td></tr>
                        <tr><th>Residence Name</th><td><?= showVal($request['residence_name']) ?></td></tr>
                        <tr><th>Floor</th><td><?= showVal($request['floor']) ?></td></tr>
                        <tr><th>Tower Number</th><td><?= showVal($request['tower_number']) ?></td></tr>
                        <tr><th>Monthly Installment</th><td><?= moneyVal($request['monthly_installment']) ?></td></tr>
                        <tr><th>Created At</th><td><?= h($request['created_at']) ?></td></tr>
                    </tbody>
                </table>
            </div>

            <div class="mt-3 small text-muted">
                <?php
                $t = $request['property_type'];
                if ($t === 'house') {
                    echo "Tip: Unit & Builded Year often matter most for houses.";
                } elseif ($t === 'land') {
                    echo "Tip: Land Number & exact Area are key for lands.";
                } elseif ($t === 'apartment_installment') {
                    echo "Tip: Residence, Floor, Tower & Monthly Installment are key for installment apartments.";
                } elseif ($t === 'apartment_cash') {
                    echo "Tip: Residence & Price are key for cash apartments.";
                }
                ?>
            </div>
        </div>
    </div>

    <!-- Related Properties (closest match first) -->
    <div class="d-flex align-items-center justify-content-between mb-2 mt-2">
        <h4 class="mb-0">Related Properties</h4>
    </div>

    <?php
    // ---------- Tight matching ----------
    $mappedType = normTypeForProperties($request['property_type']);

    // Normalize request values
    $area_req         = trim((string)($request['area'] ?? ''));
    $street_req       = trim((string)($request['street'] ?? ''));
    $size_req         = is_numeric($request['size_meters'] ?? null) ? (float)$request['size_meters'] : null;
    $price_req        = is_numeric($request['price'] ?? null) ? (float)$request['price'] : null;
    $directions_req   = trim((string)($request['directions'] ?? ''));
    $residence_req    = trim((string)($request['residence_name'] ?? ''));
    $land_number_req  = trim((string)($request['land_number'] ?? ''));

    // Dynamic windows
    $sizeTol = $minSize = $maxSize = null;
    if ($size_req !== null && $size_req > 0) {
        // ± max(20 m², 12%)
        $sizeTol = max(20, round($size_req * 0.12));
        $minSize = max(0, $size_req - $sizeTol);
        $maxSize = $size_req + $sizeTol;
    }

    $priceTol = $minPrice = $maxPrice = null;
    if ($price_req !== null && $price_req > 0) {
        // ± max($5k, 10%)
        $priceTol = max(5000, round($price_req * 0.10));
        $minPrice = max(0, $price_req - $priceTol);
        $maxPrice = $price_req + $priceTol;
    }

    $params = [ ':type' => $mappedType ];
    $where  = ["p.property_type = :type"];

    // Keep candidate set relevant
    if ($area_req !== '') {
        $where[] = "p.area IS NOT NULL AND p.area <> ''";
    }
    if ($residence_req !== '') {
        $where[] = "(p.residence_name IS NOT NULL AND p.residence_name <> '')";
    }
    if ($street_req !== '') {
        $where[] = "(p.street_address IS NOT NULL AND p.street_address <> '')";
    }
    if ($directions_req !== '') {
        $where[] = "(p.directions IS NOT NULL AND p.directions <> '')";
    }
    if ($land_number_req !== '') {
        $where[] = "(p.land_number IS NOT NULL AND p.land_number <> '')";
    }
    if ($sizeTol !== null) {
        $where[] = "p.size_sqft BETWEEN :minSize AND :maxSize";
        $params[':minSize'] = $minSize;
        $params[':maxSize'] = $maxSize;
    }
    if ($priceTol !== null) {
        $where[] = "p.price BETWEEN :minPrice AND :maxPrice";
        $params[':minPrice'] = $minPrice;
        $params[':maxPrice'] = $maxPrice;
    }

    // Scoring: exact signals first, then nearest size/price, then newest
    $sql = "
    SELECT
      p.id, p.street_address, p.price, p.size_sqft, p.area, p.residence_name, p.directions, p.land_number,

      -- exactness scores
      (CASE WHEN :area IS NOT NULL AND :area <> '' AND LOWER(p.area) = :area THEN 2
            WHEN :area_like IS NOT NULL AND LOWER(p.area) LIKE :area_like THEN 1 ELSE 0 END) AS area_score,

      (CASE WHEN :residence_req IS NOT NULL AND :residence_req <> '' AND LOWER(COALESCE(p.residence_name,'')) = :residence_req THEN 1 ELSE 0 END) AS residence_score,

      (CASE WHEN :street_req IS NOT NULL AND :street_req <> '' AND LOWER(COALESCE(p.street_address,'')) = :street_req THEN 1 ELSE 0 END) AS street_score,

      (CASE WHEN :dir_req IS NOT NULL AND :dir_req <> '' AND LOWER(COALESCE(p.directions,'')) = :dir_req THEN 1 ELSE 0 END) AS dir_score,

      (CASE WHEN :land_num_req IS NOT NULL AND :land_num_req <> '' AND LOWER(COALESCE(p.land_number,'')) = :land_num_req THEN 1 ELSE 0 END) AS landnum_score,

      -- proximity (lower is better)
      (CASE WHEN :size IS NULL  THEN 999999 ELSE ABS(p.size_sqft - :size) END)  AS size_diff,
      (CASE WHEN :price IS NULL THEN 999999 ELSE ABS(p.price - :price) END)    AS price_diff

    FROM properties p
    " . ($where ? "WHERE " . implode(" AND ", $where) : "") . "
    ORDER BY
      (area_score + residence_score + street_score + dir_score + landnum_score) DESC,
      size_diff ASC,
      price_diff ASC,
      p.id DESC
    LIMIT 24
    ";

    $matchStmt = $pdo->prepare($sql);

    // Scalars referenced in SELECT (bind even if WHERE didn't)
    $scalars = [
      ':area'          => ($area_req !== '' ? mb_strtolower($area_req, 'UTF-8') : null),
      ':area_like'     => ($area_req !== '' ? '%'.mb_strtolower($area_req, 'UTF-8').'%' : null),
      ':residence_req' => ($residence_req !== '' ? mb_strtolower($residence_req, 'UTF-8') : null),
      ':street_req'    => ($street_req !== '' ? mb_strtolower($street_req, 'UTF-8') : null),
      ':dir_req'       => ($directions_req !== '' ? mb_strtolower($directions_req, 'UTF-8') : null),
      ':land_num_req'  => ($land_number_req !== '' ? mb_strtolower($land_number_req, 'UTF-8') : null),
      ':size'          => ($size_req !== null ? $size_req : null),
      ':price'         => ($price_req !== null ? $price_req : null),
    ];
    foreach ($scalars as $k => $v) $matchStmt->bindValue($k, $v);

    // WHERE params
    foreach ($params as $k => $v) {
        if (!array_key_exists($k, $scalars)) $matchStmt->bindValue($k, $v);
    }

    $matchStmt->execute();
    $matches = $matchStmt->fetchAll(PDO::FETCH_ASSOC);
    ?>

    <?php if (empty($matches)): ?>
        <div class="alert alert-light border">No matching properties found.</div>
    <?php else: ?>
        <div class="row">
            <?php foreach ($matches as $prop): ?>
                <div class="col-md-6 col-lg-4 mb-4">
                    <div class="card h-100 shadow-sm">
                        <?php
                        $imgStmt = $pdo->prepare("SELECT image_url FROM galleries WHERE property_id = ? LIMIT 1");
                        $imgStmt->execute([$prop['id']]);
                        $image = $imgStmt->fetchColumn();
                        $imageUrl = $image
                            ? "../assets/uploads/properties/galleries/" . $image
                            : "/assets/img/no-image.jpg";
                        ?>
                        <img src="<?= h($imageUrl) ?>" class="card-img-top" alt="Property Image" style="height:200px;object-fit:cover;">
                        <div class="card-body">
                            <h6 class="card-title mb-1 text-truncate">
                                <?= h($prop['street_address'] ?: ('Property #'.$prop['id'])) ?>
                            </h6>
                            <div class="small text-muted mb-2">
                                <?= h($prop['area'] ?: '—') ?> • <?= h((string)$prop['size_sqft']) ?> m²
                            </div>
                            <div class="fw-semibold"><?= $prop['price'] === null ? '—' : '$'.number_format((float)$prop['price']) ?></div>
                            <a href="<?= URL('property-details.php?id=' . (int)$prop['id']) ?>" class="btn btn-sm btn-outline-primary mt-2">
                                View
                            </a>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    <?php endif; ?>

</div>

<?php require_once "../includes/footer.php"; ?>
